<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

error_log("KYC Request received. Method: " . $_SERVER['REQUEST_METHOD']);
error_log("Content-Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'Not set'));

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') exit(0);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// DB setup
$host = 'localhost';
$dbname = 'startmine_db';
$username = 'startmine_db';
$password = 'startmine_db';

$conn = new mysqli($host, $username, $password, $dbname);
if ($conn->connect_error) {
    error_log("DB connection failed: " . $conn->connect_error);
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

error_log("Database connection successful.");

// Check POST
if (empty($_POST)) {
    error_log("No POST data received.");
    http_response_code(400);
    echo json_encode(['error' => 'No form data received.']);
    exit;
}

$input = $_POST;
error_log("Received POST data: " . print_r($input, true));

// Required fields
$requiredFields = [
    'firstName', 'lastName', 'email', 'phone', 'dateOfBirth', 'nationality',
    'street', 'city', 'state', 'zipCode', 'country', 'occupation', 'sourceOfFunds',
    'idType', 'idNumber', 'idExpiry', 'termsAccepted', 'privacyAccepted'
];

$missingFields = [];
foreach ($requiredFields as $field) {
    if (!isset($input[$field]) || trim($input[$field]) === '') {
        $missingFields[] = $field;
    }
}

if ($input['nationality'] === 'us' && (!isset($input['ssn']) || trim($input['ssn']) === '')) {
    $missingFields[] = 'ssn';
} elseif ($input['nationality'] === 'gb' && (!isset($input['nationalInsuranceNumber']) || trim($input['nationalInsuranceNumber']) === '')) {
    $missingFields[] = 'nationalInsuranceNumber';
}

$requiredFiles = ['idFrontFile', 'idBackFile', 'selfieFile'];
error_log("Received FILES data: " . print_r($_FILES, true));
foreach ($requiredFiles as $file) {
    if (!isset($_FILES[$file]) || $_FILES[$file]['error'] !== UPLOAD_ERR_OK) {
        $missingFields[] = $file;
    }
}

if (!empty($missingFields)) {
    error_log("Missing required fields/files: " . implode(', ', $missingFields));
    http_response_code(400);
    echo json_encode([
        'error' => 'Missing required fields or files',
        'fields' => $missingFields
    ]);
    exit;
}

if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
    error_log("Invalid email: " . $input['email']);
    http_response_code(400);
    echo json_encode(['error' => 'Invalid email format']);
    exit;
}

$dob = new DateTime($input['dateOfBirth']);
$today = new DateTime();
$age = $today->diff($dob)->y;

$expiry = new DateTime($input['idExpiry']);
if ($expiry < $today) {
    error_log("ID expired on " . $input['idExpiry']);
    http_response_code(400);
    echo json_encode(['error' => 'Document must not be expired']);
    exit;
}

// Check existing email
$stmt = $conn->prepare("SELECT id FROM kyc_applications WHERE email = ?");
$stmt->bind_param("s", $input['email']);
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows > 0) {
    error_log("Email already exists: " . $input['email']);
    http_response_code(409);
    echo json_encode(['error' => 'Email already registered']);
    $stmt->close();
    $conn->close();
    exit;
}
$stmt->close();

// Prepare data
$data = [
    'first_name' => trim($input['firstName']),
    'last_name' => trim($input['lastName']),
    'email' => strtolower(trim($input['email'])),
    'phone' => trim($input['phone']),
    'date_of_birth' => $input['dateOfBirth'],
    'nationality' => $input['nationality'],
    'ssn' => $input['nationality'] === 'us' ? trim($input['ssn'] ?? '') : '',
    'national_insurance_number' => $input['nationality'] === 'gb' ? trim($input['nationalInsuranceNumber'] ?? '') : '',
    'street' => trim($input['street']),
    'city' => trim($input['city']),
    'state' => trim($input['state']),
    'zip_code' => trim($input['zipCode']),
    'country' => $input['country'] ?? '',
    'occupation' => trim($input['occupation']),
    'source_of_funds' => $input['sourceOfFunds'],
    'expected_transaction_volume' => $input['expectedTransactionVolume'] ?? '',
    'id_type' => $input['idType'],
    'id_number' => trim($input['idNumber']),
    'id_expiry' => $input['idExpiry'],
    'terms_accepted' => $input['termsAccepted'] === 'true' ? 1 : 0,
    'privacy_accepted' => $input['privacyAccepted'] === 'true' ? 1 : 0,
    'marketing_accepted' => $input['marketingAccepted'] === 'true' ? 1 : 0,
    'status' => 'pending',
    'created_at' => date('Y-m-d H:i:s'),
    'updated_at' => date('Y-m-d H:i:s')
];

// Upload files
$uploadDir = '/home/startmine/public_html/kyc/uploads/';
if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
    error_log("Upload dir not usable: $uploadDir");
    http_response_code(500);
    echo json_encode(['error' => 'Server upload directory issue.']);
    $conn->close();
    exit;
}

$filePaths = [];
foreach ($requiredFiles as $file) {
    $fileName = uniqid() . '_' . basename($_FILES[$file]['name']);
    $uploadFile = $uploadDir . $fileName;
    if (move_uploaded_file($_FILES[$file]['tmp_name'], $uploadFile)) {
        $filePaths[$file] = $uploadFile;
        error_log("Uploaded {$file} to {$uploadFile}");
    } else {
        error_log("Failed to upload {$file}");
        http_response_code(500);
        echo json_encode(['error' => "Failed to upload file: {$file}"]);
        $conn->close();
        exit;
    }
}

$data['id_front_file_path'] = $filePaths['idFrontFile'];
$data['id_back_file_path'] = $filePaths['idBackFile'];
$data['selfie_file_path'] = $filePaths['selfieFile'];

try {
    $sql = "INSERT INTO kyc_applications (
        first_name, last_name, email, phone, date_of_birth, nationality, ssn, national_insurance_number,
        street, city, state, zip_code, country, occupation, source_of_funds,
        expected_transaction_volume, id_type, id_number, id_expiry,
        terms_accepted, privacy_accepted, marketing_accepted,
        id_front_file_path, id_back_file_path, selfie_file_path,
        status, created_at, updated_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    if (count($data) !== 28) {
        error_log("Data array count mismatch. Expected 28, got " . count($data));
    }

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }

    $stmt->bind_param("ssssssssssssssssssssssssssss",
        $data['first_name'], $data['last_name'], $data['email'], $data['phone'], $data['date_of_birth'], $data['nationality'], $data['ssn'], $data['national_insurance_number'],
        $data['street'], $data['city'], $data['state'], $data['zip_code'], $data['country'], $data['occupation'], $data['source_of_funds'],
        $data['expected_transaction_volume'], $data['id_type'], $data['id_number'], $data['id_expiry'],
        $data['terms_accepted'], $data['privacy_accepted'], $data['marketing_accepted'],
        $data['id_front_file_path'], $data['id_back_file_path'], $data['selfie_file_path'],
        $data['status'], $data['created_at'], $data['updated_at']
    );

    if (!$stmt->execute()) {
        throw new Exception("Execute failed: " . $stmt->error);
    }

    $applicationId = $conn->insert_id;
    error_log("KYC Application submitted: ID $applicationId, Email: " . $data['email']);

    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'KYC application submitted successfully',
        'application_id' => $applicationId,
        'status' => 'pending'
    ]);

} catch (Exception $e) {
    error_log("KYC DB Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Failed to process application']);
} finally {
    $conn->close();
}
?>
